% Copyright 2014 Sandia Corporation. Under the terms of Contract DE-AC04-94AL85000 with Sandia
% Corporation, the U.S. Government retains certain rights in this software
%
% This file is part of Sandia SPT (Sandia Simple Particle Tracking) v. 1.0.
% 
% Sandia SPT is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% Sandia SPT is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with Sandia SPT.  If not, see <http://www.gnu.org/licenses/>.

function positions = singleMoleculeAnalysis(directory,threshold,display)

%function positions = singleMoleculeAnalysis(directory,threshold,display)
%
%PURPOSE:
%	singleMoleculeAnalysis.m is designed to analyze a directory which
%	contains (named in sorted order) .tiff images forming a movie. The code
%	will read each image, detect where there are signals using a simple
%	local maxima and thresholding scheme, and then perform subpixel
%	localization upon each signal. 
%
%DEPENDENCIES:
%	findExtrema.m
%	robustStandardDeviation.m
%	gauss2dcirc.m
%
%INPUTS:
%	directory:
%		The directory to be analyzed, which should contain one file per
%		frame, named such that they appear sequentially in the directory. 
%	threshold:
%		Only regional maxima pixels whose intensity is greater than this
%		threshold value will be considered as signals. 
%	display:
%		A logical (true or false) value, indicating whether the code should
%		pause after each frame (true) displaying a grayscale image of the
%		frame with the localizations of the detected signals indicated by
%		superimposed red circles. 
%
%OUTPUTS:
%	positions:
%		Each row corresponds to a signal detected in a frame. The first
%		two columns correspond to the x and y coordinates respectively 
%		(in units of pixels). The third column corresponds to the frame
%		number in which that signal was detected. 
%
%LIMITATIONS:
%	The code assumes that the files are arranged in the directory such that
%	they will be read in the proper order. The code uses a hard-coded
%	fittingWidth, which will work only for a specific range of point spread
%	function widths. The window selected, +/-fittingWidth pixels from the
%	brightest pixel, should be wide enough to contain the vast majority of
%	the point spread function such that at least some of the pixels at the
%	edges of the window should be back to very near background levels. 
%
%NOTES:
%	Maxima detected within fittingWidth pixels of the edge are discarded 
%	prior to 2D-Gaussian fitting. These positions are too close to the 
%	edge, and the subregion of the image used for 2D-Gaussian fitting 
%	would extend beyond the edge of the image. Sub-pixel localization, 
%	however, can result in positions within the earlier excluded region. 
%
%Written by Stephen M. Anthony 08/19/2014
%Last modified on 08/19/14 by Stephen M. Anthony

%Determines all files in the directory
files = dir(directory);

%Removes the first two entries, which are always included and correspond to
%the present directory and parent directory
files = files(3:end);

%Determine the number of files
nFiles = numel(files);

%Initialize the kernel for detecting local maxima
kernel = ones(5);

%Specify the number of pixels out from the center to use.
fittingWidth = 2;

%Initialize the offsets for later data handling
range = -fittingWidth:fittingWidth;
[xOffsets,yOffsets] = meshgrid(range,range);

%Initialize the output positions
positions = cell(nFiles,1);

%Code for progress display
digits = ceil(log10(nFiles));
numberFormat = ['%0' num2str(digits) 'd'];
backspace = {'\b'};
undo = cell2mat(backspace(ones(1,digits+1)));
fprintf(['There are ' num2str(nFiles) ' files to process.\n']);
fprintf(['Currently working on file #' num2str(0,numberFormat)]);

for frameIndex=1:nFiles
	%Display the progress. 
	fprintf([undo numberFormat '.'],frameIndex);
	
	%Read in the specified frame
	image = imread(fullfile(directory,files(frameIndex).name));
	image =  double(image);
	
	%Find those pixels in the image that are at least as large as all
	%neighboring pixels and pixels 2 away. 
	[~,maxima] = findExtrema(image,kernel,false);
	
	%Find those pixels which are maxima and above the threshold
	[row,col] = find(maxima & (image>threshold));
	
	%Eliminate those detections which are two near the edge, such that
	%further analysis would require pixels off the edge of the image. 
	nearE = row<fittingWidth+1 | row>size(image,1)-fittingWidth ...
		| col<fittingWidth+1 | col>size(image,2)-fittingWidth;
	col = col(~nearE);
	row = row(~nearE);
	
	%Subtract the median as a rough background subtraction
	image = image-median(image(:));
	
	%Estimate the noise level of the background
	sigmaEst = robustStandardDeviation(image);
	
	%Initialize
	nDetections = numel(col);
	subPix = nan(nDetections,2);
	
	%Loop through all detections to obtain their subpixel localizations
	for detectionIndex=1:nDetections
		%Determine the x and y coordinates the pixels to be analyzed (those
		%pixels +/- 2 pixels from the maximum found earlier). 
		x = xOffsets(:)+col(detectionIndex);
		y = yOffsets(:)+row(detectionIndex);
		
		%Determine the intensity (z) of each of the pixels used. 
		lindex = sub2ind(size(image),y,x);
		z = image(lindex);
		
		%Perform a rapid 2D-Gaussian fitting
		[xc,yc]=gauss2dcirc(z,x,y,sigmaEst);
		
		%Record the sub-pixel localization for this detection
		subPix(detectionIndex,:) = [xc yc];
	end

	%Record the positions from this frame
	positions{frameIndex} = [subPix ones(nDetections,1)*frameIndex];
	
	%Optional, displays the image and the localizations. 
	if exist('display','var') && display
		imagesc(image);
		set(gca,'DataAspectRatio',[1 1 1])
		colormap('gray')
		hold on;plot(subPix(:,1),subPix(:,2),'ro','LineWidth',2);hold off
		pause
	end
end

%Convert to a matrix. 
positions = cell2mat(positions);

%Display done
fprintf('\n');
disp('Done');